<?php
use Livewire\Volt\Component;
use Livewire\WithPagination;
use App\Models\Z_Website\Market\MarketTransaction;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

new class extends Component {
    use WithPagination;

    public $search = '';
    public $transactionType = 'all';
    public $currencyType = 'all';
    public $status = 'all';
    public $dateFilter = 'all';
    public $customDateFrom = '';
    public $customDateTo = '';
    public $userFilter = '';
    public $perPage = 25;

    // Statistics
    public $stats = [];
    public $chartData = [];

    public function mount()
    {
        $this->loadStats();
        $this->customDateFrom = now()->startOfMonth()->format('Y-m-d');
        $this->customDateTo = now()->format('Y-m-d');
    }

    public function loadStats()
    {
        $baseQuery = MarketTransaction::query();

        $this->stats = [
            'total_transactions' => (clone $baseQuery)->count(),
            'successful_transactions' => (clone $baseQuery)->success()->count(),
            'failed_transactions' => (clone $baseQuery)->failed()->count(),
            'pending_transactions' => (clone $baseQuery)->pending()->count(),
            'today_transactions' => (clone $baseQuery)->today()->count(),
            'this_week_transactions' => (clone $baseQuery)->thisWeek()->count(),
            'this_month_transactions' => (clone $baseQuery)->thisMonth()->count(),
            'total_revenue' => (clone $baseQuery)->success()->sum('total_price'),
            'avg_transaction_value' => (clone $baseQuery)->success()->avg('total_price'),
        ];

        // Chart data for last 7 days
        $this->chartData = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $this->chartData[] = [
                'date' => $date->format('M d'),
                'transactions' => MarketTransaction::whereDate('created_at', $date)->count(),
                'revenue' => MarketTransaction::whereDate('created_at', $date)->success()->sum('total_price')
            ];
        }
    }

    public function getTransactionsProperty()
    {
        $query = MarketTransaction::with('user')
            ->orderBy('created_at', 'desc');

        // Apply filters
        if (!empty($this->search)) {
            $query->where(function ($q) {
                $q->where('transaction_id', 'like', '%' . $this->search . '%')
                  ->orWhere('username', 'like', '%' . $this->search . '%')
                  ->orWhere('item_name', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->transactionType !== 'all') {
            $query->where('transaction_type', $this->transactionType);
        }

        if ($this->currencyType !== 'all') {
            $query->where('currency_type', $this->currencyType);
        }

        if ($this->status !== 'all') {
            $query->where('status', $this->status);
        }

        if (!empty($this->userFilter)) {
            $query->where(function ($q) {
                $q->where('user_jid', $this->userFilter)
                  ->orWhere('username', 'like', '%' . $this->userFilter . '%');
            });
        }

        // Date filter
        switch ($this->dateFilter) {
            case 'today':
                $query->today();
                break;
            case 'week':
                $query->thisWeek();
                break;
            case 'month':
                $query->thisMonth();
                break;
            case 'custom':
                if ($this->customDateFrom && $this->customDateTo) {
                    $query->whereBetween('created_at', [
                        Carbon::parse($this->customDateFrom)->startOfDay(),
                        Carbon::parse($this->customDateTo)->endOfDay()
                    ]);
                }
                break;
        }

        return $query->paginate($this->perPage);
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedTransactionType()
    {
        $this->resetPage();
    }

    public function updatedCurrencyType()
    {
        $this->resetPage();
    }

    public function updatedStatus()
    {
        $this->resetPage();
    }

    public function updatedDateFilter()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedCustomDateFrom()
    {
        $this->resetPage();
    }

    public function updatedCustomDateTo()
    {
        $this->resetPage();
    }

    public function updatedUserFilter()
    {
        $this->resetPage();
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->transactionType = 'all';
        $this->currencyType = 'all';
        $this->status = 'all';
        $this->dateFilter = 'all';
        $this->userFilter = '';
        $this->customDateFrom = now()->startOfMonth()->format('Y-m-d');
        $this->customDateTo = now()->format('Y-m-d');
        $this->resetPage();
        $this->loadStats();
    }

    public function exportTransactions()
    {
        // Export functionality would go here
        $this->dispatch('show-notification', [
            'type' => 'info',
            'message' => 'Export feature will be implemented soon!',
            'duration' => 3000
        ]);
    }

    public function viewTransactionDetails($transactionId)
    {
        $this->dispatch('show-transaction-modal', ['transactionId' => $transactionId]);
    }
};
?>

<div>
    <div class="space-y-8" x-data="{
        showDetailsModal: false,
        selectedTransaction: null,
        chartData: @js($this->chartData)
    }">

        <!-- Page Header -->
        <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
            <div>
                <h1 class="text-4xl font-bold text-white mb-3 dashboard-gaming-text">
                    <i class="fas fa-chart-line mr-3 text-purple-400"></i>
                    Market Transaction Logs
                </h1>
                <p class="text-xl text-purple-300">
                    Monitor and analyze all market transactions
                </p>
            </div>
            <div class="flex flex-col sm:flex-row gap-3 mt-4 lg:mt-0">
                <button wire:click="loadStats" class="btn-secondary">
                    <i class="fas fa-sync mr-2"></i>
                    Refresh Stats
                </button>
                <button wire:click="exportTransactions" class="btn-primary">
                    <i class="fas fa-download mr-2"></i>
                    Export Data
                </button>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-5 gap-4 mb-8">
            <!-- Total Transactions -->
            <div class="bg-gradient-to-br from-blue-900/50 to-blue-800/30 rounded-xl p-4 border border-blue-500/30">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-blue-300 text-sm">Total Transactions</p>
                        <p class="text-white text-2xl font-bold">{{ number_format($stats['total_transactions']) }}</p>
                    </div>
                    <i class="fas fa-exchange-alt text-blue-400 text-2xl"></i>
                </div>
            </div>

            <!-- Successful -->
            <div class="bg-gradient-to-br from-green-900/50 to-green-800/30 rounded-xl p-4 border border-green-500/30">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-green-300 text-sm">Successful</p>
                        <p class="text-white text-2xl font-bold">{{ number_format($stats['successful_transactions']) }}</p>
                    </div>
                    <i class="fas fa-check-circle text-green-400 text-2xl"></i>
                </div>
            </div>

            <!-- Failed -->
            <div class="bg-gradient-to-br from-red-900/50 to-red-800/30 rounded-xl p-4 border border-red-500/30">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-red-300 text-sm">Failed</p>
                        <p class="text-white text-2xl font-bold">{{ number_format($stats['failed_transactions']) }}</p>
                    </div>
                    <i class="fas fa-times-circle text-red-400 text-2xl"></i>
                </div>
            </div>

            <!-- Today -->
            <div class="bg-gradient-to-br from-purple-900/50 to-purple-800/30 rounded-xl p-4 border border-purple-500/30">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-purple-300 text-sm">Today</p>
                        <p class="text-white text-2xl font-bold">{{ number_format($stats['today_transactions']) }}</p>
                    </div>
                    <i class="fas fa-calendar-day text-purple-400 text-2xl"></i>
                </div>
            </div>

            <!-- Total Revenue -->
            <div class="bg-gradient-to-br from-yellow-900/50 to-yellow-800/30 rounded-xl p-4 border border-yellow-500/30">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-yellow-300 text-sm">Total Revenue</p>
                        <p class="text-white text-xl font-bold">{{ number_format($stats['total_revenue'], 0) }}</p>
                    </div>
                    <i class="fas fa-coins text-yellow-400 text-2xl"></i>
                </div>
            </div>
        </div>

        <!-- Chart Section -->
        <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-2xl p-6 border border-purple-500/30 mb-8">
            <h3 class="text-xl font-bold text-white mb-4">
                <i class="fas fa-chart-area mr-2 text-cyan-400"></i>
                Last 7 Days Activity
            </h3>
            <div class="h-64 flex items-end justify-between space-x-2">
                @foreach($chartData as $day)
                    <div class="flex-1 flex flex-col items-center">
                        <div class="w-full bg-gradient-to-t from-cyan-600 to-cyan-400 rounded-t-lg relative group cursor-pointer"
                             style="height: {{ $day['transactions'] > 0 ? max(20, ($day['transactions'] / max(array_column($chartData, 'transactions'))) * 200) : 4 }}px;"
                             title="Transactions: {{ $day['transactions'] }}, Revenue: {{ number_format($day['revenue']) }}">
                            <div class="absolute -top-8 left-1/2 transform -translate-x-1/2 bg-gray-800 text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap">
                                {{ $day['transactions'] }} txns<br>
                                {{ number_format($day['revenue']) }} revenue
                            </div>
                        </div>
                        <p class="text-gray-400 text-xs mt-2 text-center">{{ $day['date'] }}</p>
                    </div>
                @endforeach
            </div>
        </div>

        <!-- Filters Section -->
        <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-2xl p-6 border border-purple-500/30 mb-8">
            <h3 class="text-lg font-bold text-white mb-4">
                <i class="fas fa-filter mr-2 text-purple-400"></i>
                Filters
            </h3>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-4">
                <!-- Search -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">Search</label>
                    <input type="text" wire:model.live="search"
                           class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500"
                           placeholder="Transaction ID, username, item...">
                </div>

                <!-- Transaction Type -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">Transaction Type</label>
                    <select wire:model.live="transactionType"
                            class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                        <option value="all">All Types</option>
                        <option value="package">Packages</option>
                        <option value="item">Items</option>
                    </select>
                </div>

                <!-- Currency Type -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">Currency</label>
                    <select wire:model.live="currencyType"
                            class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                        <option value="all">All Currencies</option>
                        <option value="balance">💎 Balance</option>
                        <option value="gift_balance">⭐ Gift Balance</option>
                        <option value="silk">✨ Silk</option>
                        <option value="donate">💖 Donate</option>
                    </select>
                </div>

                <!-- Status -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">Status</label>
                    <select wire:model.live="status"
                            class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                        <option value="all">All Status</option>
                        <option value="success">✅ Success</option>
                        <option value="failed">❌ Failed</option>
                        <option value="pending">⏳ Pending</option>
                    </select>
                </div>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <!-- Date Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">Date Range</label>
                    <select wire:model.live="dateFilter"
                            class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                        <option value="all">All Time</option>
                        <option value="today">Today</option>
                        <option value="week">This Week</option>
                        <option value="month">This Month</option>
                        <option value="custom">Custom Range</option>
                    </select>
                </div>

                @if($dateFilter === 'custom')
                    <!-- From Date -->
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">From Date</label>
                        <input type="date" wire:model.live="customDateFrom"
                               class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                    </div>

                    <!-- To Date -->
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">To Date</label>
                        <input type="date" wire:model.live="customDateTo"
                               class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500">
                    </div>
                @endif

                <!-- User Filter -->
                <div>
                    <label class="block text-sm font-medium text-gray-300 mb-2">User Filter</label>
                    <input type="text" wire:model.live="userFilter"
                           class="w-full bg-gray-800 border border-gray-600 rounded-lg px-3 py-2 text-white focus:border-purple-500 focus:ring-1 focus:ring-purple-500"
                           placeholder="User ID or username">
                </div>

                <!-- Reset Button -->
                <div class="flex items-end">
                    <button wire:click="resetFilters"
                            class="w-full bg-red-600 hover:bg-red-700 text-white font-bold py-2 px-4 rounded-lg transition-colors">
                        <i class="fas fa-undo mr-2"></i>
                        Reset Filters
                    </button>
                </div>
            </div>
        </div>

        <!-- Transactions Table -->
        <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-2xl border border-purple-500/30 overflow-hidden">
            <div class="p-6 border-b border-gray-700">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                    <h3 class="text-lg font-bold text-white">
                        <i class="fas fa-list mr-2 text-cyan-400"></i>
                        Transaction History
                    </h3>
                    <div class="flex items-center space-x-4 mt-4 sm:mt-0">
                        <span class="text-gray-400 text-sm">
                            Showing {{ $this->transactions->count() }} of {{ $this->transactions->total() }} transactions
                        </span>
                        <select wire:model.live="perPage"
                                class="bg-gray-800 border border-gray-600 rounded px-3 py-1 text-white text-sm">
                            <option value="10">10 per page</option>
                            <option value="25">25 per page</option>
                            <option value="50">50 per page</option>
                            <option value="100">100 per page</option>
                        </select>
                    </div>
                </div>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-800">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Transaction</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">User</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Item/Package</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Amount</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700">
                        @forelse($this->transactions as $transaction)
                            <tr class="hover:bg-gray-800/50 transition-colors">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex flex-col">
                                        <div class="text-sm font-bold text-white">{{ $transaction->transaction_id }}</div>
                                        <div class="text-xs text-purple-300">
                                            <i class="fas {{ $transaction->transaction_type === 'package' ? 'fa-box' : 'fa-gem' }} mr-1"></i>
                                            {{ ucfirst($transaction->transaction_type) }}
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex flex-col">
                                        <div class="text-sm font-medium text-white">{{ $transaction->username ?? 'N/A' }}</div>
                                        <div class="text-xs text-gray-400">ID: {{ $transaction->user_jid }}</div>
                                    </div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="flex flex-col">
                                        <div class="text-sm font-medium text-white truncate max-w-32">{{ $transaction->item_name }}</div>
                                        <div class="text-xs text-cyan-300">Qty: {{ $transaction->quantity }}</div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex flex-col">
                                        <div class="text-sm font-bold text-yellow-400">
                                            {{ $transaction->currency_symbol }} {{ number_format($transaction->total_price) }}
                                        </div>
                                        <div class="text-xs text-gray-400">
                                            {{ number_format($transaction->unit_price) }} each
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-semibold rounded-full {{ $transaction->status_badge }} text-white">
                                        {{ ucfirst($transaction->status) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-300">{{ $transaction->created_at->format('M d, Y') }}</div>
                                    <div class="text-xs text-gray-500">{{ $transaction->created_at->format('H:i:s') }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <button @click="selectedTransaction = @json($transaction); showDetailsModal = true"
                                            class="text-cyan-400 hover:text-cyan-300 transition-colors">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="px-6 py-8 text-center text-gray-400">
                                    <i class="fas fa-inbox text-4xl mb-4"></i>
                                    <p class="text-lg">No transactions found</p>
                                    <p class="text-sm">Try adjusting your filters</p>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            @if($this->transactions->hasPages())
                <div class="px-6 py-4 border-t border-gray-700">
                    {{ $this->transactions->links() }}
                </div>
            @endif
        </div>

        <!-- Transaction Details Modal -->
        <div x-show="showDetailsModal"
             x-transition:enter="transition ease-out duration-300"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition ease-in duration-200"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             class="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4"
             @click.self="showDetailsModal = false">

            <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-2xl max-w-2xl w-full max-h-[80vh] overflow-hidden border border-purple-500/30"
                 x-transition:enter="transition ease-out duration-300 transform"
                 x-transition:enter-start="scale-95 opacity-0"
                 x-transition:enter-end="scale-100 opacity-100"
                 x-transition:leave="transition ease-in duration-200 transform"
                 x-transition:leave-start="scale-100 opacity-100"
                 x-transition:leave-end="scale-95 opacity-0">

                <div class="p-6 border-b border-gray-700">
                    <div class="flex items-center justify-between">
                        <h3 class="text-xl font-bold text-white">
                            <i class="fas fa-receipt mr-2 text-cyan-400"></i>
                            Transaction Details
                        </h3>
                        <button @click="showDetailsModal = false"
                                class="text-gray-400 hover:text-white transition-colors">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>
                </div>

                <div class="p-6 overflow-y-auto max-h-[60vh]" x-show="selectedTransaction">
                    <div class="space-y-4">
                        <!-- Basic Info -->
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Transaction ID</label>
                                <p class="text-white font-mono" x-text="selectedTransaction?.transaction_id"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Status</label>
                                <span class="px-2 py-1 text-xs font-semibold rounded-full"
                                      :class="selectedTransaction?.status === 'success' ? 'bg-green-600' : selectedTransaction?.status === 'failed' ? 'bg-red-600' : 'bg-yellow-600'"
                                      x-text="selectedTransaction?.status"></span>
                            </div>
                        </div>

                        <!-- User Info -->
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Username</label>
                                <p class="text-white" x-text="selectedTransaction?.username || 'N/A'"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">User ID</label>
                                <p class="text-white font-mono" x-text="selectedTransaction?.user_jid"></p>
                            </div>
                        </div>

                        <!-- Transaction Info -->
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Type</label>
                                <p class="text-white capitalize" x-text="selectedTransaction?.transaction_type"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Currency</label>
                                <p class="text-white capitalize" x-text="selectedTransaction?.currency_type?.replace('_', ' ')"></p>
                            </div>
                        </div>

                        <!-- Item Info -->
                        <div>
                            <label class="block text-sm font-medium text-gray-400 mb-1">Item/Package Name</label>
                            <p class="text-white" x-text="selectedTransaction?.item_name"></p>
                        </div>

                        <!-- Price Info -->
                        <div class="grid grid-cols-3 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Quantity</label>
                                <p class="text-white" x-text="selectedTransaction?.quantity"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Unit Price</label>
                                <p class="text-yellow-400" x-text="selectedTransaction?.unit_price"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Total Price</label>
                                <p class="text-yellow-400 font-bold" x-text="selectedTransaction?.total_price"></p>
                            </div>
                        </div>

                        <!-- Balance Info -->
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Balance Before</label>
                                <p class="text-cyan-400" x-text="selectedTransaction?.balance_before"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Balance After</label>
                                <p class="text-cyan-400" x-text="selectedTransaction?.balance_after"></p>
                            </div>
                        </div>

                        <!-- Technical Info -->
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">IP Address</label>
                                <p class="text-white font-mono text-sm" x-text="selectedTransaction?.ip_address || 'N/A'"></p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Date</label>
                                <p class="text-white text-sm" x-text="selectedTransaction?.created_at"></p>
                            </div>
                        </div>

                        <!-- Error Message (if any) -->
                        <div x-show="selectedTransaction?.error_message">
                            <label class="block text-sm font-medium text-gray-400 mb-1">Error Message</label>
                            <p class="text-red-400 bg-red-900/20 p-3 rounded-lg text-sm" x-text="selectedTransaction?.error_message"></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
